<?php

namespace App\Http\Controllers\Admin;

use DataTables;
use PDF;
use App\Models\Sku;
use App\Models\Supplier;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Gate;


class SupplierController extends Controller
{
    public function __construct(){
        $this->middleware(['auth'])->except('supplier_quantity');
        $this->middleware('auth:supplier')->only('supplier_quantity');

        $this->middleware('permission:supplier-list')->only('index');
        $this->middleware('permission:supplier-create', ['only' => ['add','store']]);
        $this->middleware('permission:supplier-edit', ['only' => ['edit','update']]);
        $this->middleware('permission:pending-orders', ['only' => ['retrived_quantity']]);
        $this->middleware('permission:supplier-delete', ['only' => ['destroy']]);
        $this->middleware(['permission:dropin-warehouse','permission:complete-orders'], ['only' => ['status']]);
        $this->middleware('permission:print-label', ['only' => ['print_label_list']]);


    }

    public function index(){
        return view('supplier.index');
    }

    public function add(){
        return view('supplier.create');
    }

    public function allSuppliers(Request $request){
        if ($request->ajax()) {
            $records=Supplier::latest();
            return Datatables::of($records)
                ->addIndexColumn()
                ->addColumn('action', function($data){
                    $delete =  route('supplier.delete',$data->id);
                    $edit =  route('supplier.edit',$data->id);
                    if($data->id ==1){
                        $disabled='disabled';
                    }else{
                        $disabled='';
                    }

                    if(Gate::check('supplier-edit')){
                        $edit_option='<option value="edit" data-link="'.$edit.'">Edit</option>';
                    }else{
                        $edit_option='';
                    }

                    if(Gate::check('supplier-delete')){
                        $delete_option='<option value="delete" data-link="'.$delete.'">Delete</option>';
                    }else{
                        $delete_option='';
                    }
                    $options='<select name="options" id="options_'.$data->id.'" class="form-control options" onchange="get_options('.$data->id.')" '.$disabled.'>
                         <option value="">Please select</option>
                         
                         '.$edit_option.$delete_option.'
                         </select>';
                    return $options;
                })
                
                ->rawColumns(['action'])
                ->make(true);
        }
    }

    public function store(Request $request){
        $input = $request->all();

        if(!empty($input['password'])){ 
            $input['password'] = \Hash::make($input['password']);
        }else{
            $input = array_except($input,array('password'));    
        }
        
        $name=explode(' ',$input['supplier_name']);
        $short_name='';
        foreach($name as $n){
            $short_name.=$n[0];
        }
        $short_name.='S';
        
        $data=[
            'supplier_name' => $input['supplier_name'],
            'username' => $input['username'],
            'address' =>  $input['supplier_address'],
            'password' => $input['password'],
            'supplier_email' =>$input['supplier_email'],
            'supplier_contact' => $input['contact_number'],
            'sup_po_no' => 1,
            'short_name' => $short_name,
            'sup_po_no_date' => date('Y-m-d')
        ];
        Supplier::create($data);
        return redirect()->route('supplier.show')->with('has_success','Supplier  Added Successfully');

    }

    public function edit($id){
        $supplier = Supplier::find($id);
        return view('supplier.edit',['supplier' => $supplier]);
    }

    public function update($id,Request $request){
        $input = $request->all();
        $supplier = Supplier::find($id);
        $supplier->update([
            'supplier_name' => $input['supplier_name'],
            'username' => $input['username'],
            'address' =>  $input['supplier_address'],
            'supplier_email' =>$input['supplier_email'],
            'supplier_contact' => $input['contact_number']
        ]);
        return redirect()->route('supplier.show')->with('has_success','Supplier  Updated Successfully');
    }

    public function destroy($id){
        Supplier::find($id)->delete();
        return redirect()->route('supplier.show')
                        ->with('success','Supplier deleted successfully');
    }

    public function check_username(Request $request){
        $input = $request->all();
        
        $supplier=Supplier::where('username',$input['username'])->get(['username']);
        
        if(count($supplier)>0){
            $data=1;
            echo $data;

        }else{
            $data=0;
            echo $data;
            
        }

    }

    public function retrived_quantity(Request $request){
        if ($request->ajax()) {
            // dd($request->all());
            $supplier=$request->supplier_id;
            $po_id=$request->po_id;
            
            $records=\DB::table('assign_quantity')
                        ->join('suppliers','suppliers.id','=','assign_quantity.supplier_id')
                        // ->join('type_images','assign_quantity.main_type','=','type_images.code')
                        ->join('product_info','product_info.variation_id','=','assign_quantity.variation_id')
                        ->join('product_size','assign_quantity.prod_size','=','product_size.id')
                        ->select([
                            'assign_quantity.*',
                            'suppliers.supplier_name',
                            'product_info.prod_img',
                            'product_size.size',
                            'product_info.product_name',
                            'product_info.product_url'
                        ])
                        
                        ->where('assign_quantity.status',0)
                        ->where('assign_quantity.supplier_id','!=',1);
            
            if($supplier != ''){
                $records->where('assign_quantity.supplier_id',$supplier);
            }
            
            if($po_id !=NULL && count($po_id)!=0 ){
                if($po_id[0]!=''){
                    $records->whereIn('assign_quantity.po_id',$po_id);
                }
                
            }
                        
                        // dd($records);
            return Datatables::of($records)
                    ->addIndexColumn()
                    ->addColumn('remaining', function($data){
                        $sub=(int)$data->receive_qty-(int)$data->receiving_qty;
                        
                        $cancelled_btn='';
                        return $sub.$cancelled_btn;
                    })
                    ->addColumn('variation_id',function($data){
                        if($data->product_url !=''){
                            $url=$data->product_url;
                            $img=' <img src="'.$data->product_url.'" class="rounded img-thumbnail" alt="Cinque Terre" width="50" height="50">';
                        }else{
                            $url=asset('uploads/sku_images/'.$data->prod_img);
                            $img='<img src="'.asset('uploads/sku_images/'.$data->prod_img).'" class="rounded img-thumbnail" alt="Cinque Terre" width="50" height="50">';
                        }
                        
                        
                        if($data->old_assign_id !=0){
                            //assign_quantity adding new column (old_supplier_id) 
                            $switch_supplier_id=\App\Models\Supplier::findorfail($data->old_supplier_id)->supplier_name;
                            $switch_html='<h4><span class="badge badge-info">Switched Supplier : '.$switch_supplier_id.'</span></h4>';
                            
                        }else{
                             $switch_html='';
                        }  
                        
                        
                        
                
                        $product_detail='
                            <div class="row">
                                <div class="col-md-2">
                                    <a class="single_image" href="'.$url.'" target="_blank" data-featherlight="image">
                                        '.$img.'
                                    </a>
                                </div>
                                <div class="col-md-10">
                                <span class="text-success tdbreak">'.$data->product_name.'</span><br>
                                    
                                    <span class="text-primary text-sm-left">SKU : <strong>'.$data->variation_id.'</strong></span><br>
                                    <span class="text-danger text-sm-left">PO No : <strong>'.$data->po_id.'</strong></span><br>
                                    
                                    
                                    '.$switch_html.'
                                </div>
                            </div>
                        ';
                        return $product_detail;
                    })
                    ->addColumn('action',function($data){
                        if(auth()->user()->can('supplier-cancel')){
                            $cancel='<div tabindex="-1" class="dropdown-divider"></div>
                                    <button type="button" tabindex="0" class="dropdown-item cancelled_order" id="cancelled_'.$data->id.'" data-assign_id="'.$data->id.'">Cancel Order</button>';
                        }else{
                            $cancel='';
                        }
                        $html='<div class="input-group-btn">
                                    <div class="btn-group">
                                        <button type="button" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false" class="dropdown-toggle btn btn-primary">Action</button>
                                        <div tabindex="-1" aria-hidden="true" role="menu" class="dropdown-menu">
                                            <button type="button" tabindex="0" class="dropdown-item switch_supplier" id="switch_supplier_'.$data->id.'" data-assign_id="'.$data->id.'">Switch Supplier</button>
                                            '.$cancel.'
                                        </div>
                                    </div>
                                </div>';
                        return $html;
                    })
                    ->filterColumn('variation_id',function($query,$keyword){
                        $query->orWhereRaw("LOWER(product_info.product_name) LIKE ?", ["%{$keyword}%"])
                                ->orWhereRaw("assign_quantity.variation_id LIKE ?", ["%{$keyword}%"])
                                ->orWhereRaw("assign_quantity.pr_id LIKE ?", ["%{$keyword}%"])
                                ->orWhereRaw("assign_quantity.po_id LIKE ?", ["%{$keyword}%"]);
                    })
                    ->orderColumn('variation_id', function ($query, $order) {
                        $query->orderBy('product_info.product_name', $order)
                            ->orderBy('assign_quantity.variation_id', $order);    
                    })
                    
                    ->addColumn('assign_date', function($data){
                        return date('d-F-Y',strtotime($data->assign_date));
                    })
                    
                    ->filterColumn('assign_date',function($query,$keyword){
                        $keyword=date('Y-m-d',strtotime($keyword));
                        $query->orWhereRaw("assign_quantity.assign_date LIKE ?", ["%{$keyword}%"]);
                    })
                    ->orderColumn('assign_date', function ($query, $order) {
                        $query->orderBy('assign_quantity.assign_date', $order);
                    })
                    ->rawColumns(['variation_id','size','receive_qty','receiving_qty','remaining','assign_date','action'])
                    ->make(true);

        }else{
            $suppliers=Supplier::latest()->where('id','!=','2')->where('id','!=','1')->get();
            // $suppliers=Supplier::latest()->where('id','!=','2')->get();
            
            $po_list=\DB::table('assign_quantity')->select(['assign_quantity.po_id','assign_quantity.po_id','assign_quantity.assign_date','suppliers.supplier_name','purchase_order_quantity.shipment_type','store.strong_name'])
                    ->join('suppliers','suppliers.id','=','assign_quantity.supplier_id')
                    ->join('purchase_order_quantity','purchase_order_quantity.pr_id','=','assign_quantity.pr_id')
                    ->join('store','store.id','=','purchase_order_quantity.store_id')
                    ->where('assign_quantity.status',0)
                    ->where('assign_quantity.supplier_id','!=',1)
                    ->groupBy('assign_quantity.po_id')->get();
                    
            $total_po=\DB::table('assign_quantity')
                        ->select([
                                \DB::raw('sum(assign_quantity.receive_qty) as total_order,sum(assign_quantity.receiving_qty) as total_received,(sum(assign_quantity.receive_qty)-sum(assign_quantity.receiving_qty)) as remaining')
                            ])
                        ->where('assign_quantity.status',0)
                        ->where('assign_quantity.supplier_id','!=',1)
                        ->where('assign_quantity.supplier_id','!=',2)
                        ->first();
            // $total_po=0;
                                
                    
            return view('supplier.retrive',['suppliers' => $suppliers,'po_list' => $po_list,'total_po' => $total_po]);
        }
        
    }

    public function supplier_quantity(Request $request){
        if ($request->ajax()) {
            // dd($request->all());
            $supplier=auth('supplier')->user()->id;
            
            $records=\DB::table('assign_quantity')
                        ->join('suppliers','suppliers.id','=','assign_quantity.supplier_id')
                        // ->join('type_images','assign_quantity.main_type','=','type_images.code')
                        ->join('product_info','product_info.variation_id','=','assign_quantity.variation_id')
                        ->join('product_size','assign_quantity.prod_size','=','product_size.id')
                        ->select([
                                'assign_quantity.assign_date',
                                'assign_quantity.variation_id',
                                'assign_quantity.id',
                                \DB::raw('sum(receive_qty) as receive_qty'),
                                \DB::raw('sum(receiving_qty) as receiving_qty'),
                                'suppliers.supplier_name',
                                'product_info.prod_img',
                                'product_size.size',
                                'product_info.product_name',
                                'product_info.product_url'
                        ])
                        
                        ->groupby('assign_quantity.variation_id')
                        ->where('assign_quantity.supplier_id',$supplier)
                        ->where('assign_quantity.status',0);
                    // dd($records);
            
                        
            return Datatables::of($records)
                    ->addIndexColumn()
                    ->addColumn('remaining', function($data){
                        $sub=(int)$data->receive_qty-(int)$data->receiving_qty;
                        return $sub;
                    })
                    ->addColumn('variation_id',function($data){
                        if($data->product_url !=''){
                            $url=$data->product_url;
                            $img=' <img src="'.$data->product_url.'" class="rounded img-thumbnail" alt="Cinque Terre" width="100" height="100">';
                        }else{
                            $url=asset('uploads/sku_images/'.$data->prod_img);
                            $img='<img src="'.asset('uploads/sku_images/'.$data->prod_img).'" class="rounded img-thumbnail"  alt="Cinque Terre" width="100" height="100">';
                        }
                        $product_detail='
                            <div class="row">
                                <div class="col-md-2">
                                    <a class="single_image" href="'.$url.'" target="_blank" data-featherlight="image">
                                        '.$img.'
                                    </a>
                                </div>
                                <div class="col-md-10">
                                <span class="text-success">'.$data->product_name.'</span><br>
                                    <span class="text-primary text-sm-left">Variation id : <strong>'.$data->variation_id.'</strong></span><br>
                                </div>
                            </div>
                        ';
                        return $product_detail;
                    })
                    ->filterColumn('variation_id',function($query,$keyword){
                        $query->orWhereRaw("LOWER(product_info.product_name) LIKE ?", ["%{$keyword}%"])
                                ->orWhereRaw("assign_quantity.variation_id LIKE ?", ["%{$keyword}%"]);     
                    })
                    ->orderColumn('variation_id', function ($query, $order) {
                        $query->orderBy('product_info.product_name', $order)
                            ->orderBy('assign_quantity.variation_id', $order);    
                    })
                    ->addColumn('assign_date', function($data){
                        return date('d-m-Y',strtotime($data->assign_date));
                    })
                    ->filterColumn('assign_date',function($query,$keyword){
                        $keyword=date('Y-m-d',strtotime($keyword));
                        $query->orWhereRaw("assign_quantity.assign_date LIKE ?", ["%{$keyword}%"]);
                    })
                    ->orderColumn('assign_date', function ($query, $order) {
                        $query->orderBy('assign_quantity.assign_date', $order);
                    })
                    ->rawColumns(['variation_id','remaining','assign_date'])
                    ->make(true);

        }else{
            // $suppliers=Supplier::latest()->get();
            return view('supplier.quantity');
        }
        echo "hello";
        
    }

    public function status(Request $request,$string){
        if($request->ajax()){
            $supplier=$request->supplier_id;
            $po_id=$request->po_id;
            $records=\DB::table('assign_quantity')
                        // ->join('suppliers','suppliers.id','=','assign_quantity.supplier_id')
                        // ->join('type_images','assign_quantity.main_type','=','type_images.code')
                        ->join('product_info','product_info.variation_id','=','assign_quantity.variation_id')
                        ->join('product_size','product_info.prod_size','=','product_size.id')
                        // ->orderBy('assign_quantity.')
                        ->orderby('assign_quantity.id','ASC')
                        ->select([
                            'assign_quantity.*',
                            // 'suppliers.supplier_name',
                            'product_info.prod_img',
                            'product_info.product_url',
                            'product_size.size',
                            'product_info.product_name'
                        ])
                        ->where('assign_quantity.supplier_id',$supplier);
            if($request->type=='1'){
                $records->where('assign_quantity.status',0);
            }else{
                $records->where('assign_quantity.status',1);

            }
            
            if($po_id != ''){
                $records->where('assign_quantity.po_id',$po_id);
            }
            // $string
            return Datatables::of($records)
                    ->addIndexColumn()
                    ->addColumn('variation_id',function($data){
                        if($data->product_url !=''){
                            $url=$data->product_url;
                            $img=' <img src="'.$data->product_url.'" class="rounded img-thumbnail" alt="Cinque Terre" width="70" height="70">';
                        }else{
                            $url=asset('uploads/sku_images/'.$data->prod_img);
                            $img='<img src="'.asset('uploads/sku_images/'.$data->prod_img).'" class="rounded img-thumbnail" alt="Cinque Terre" width="70" height="70">';
                        }
                        
                        //added by sufian 05-01-22
                        if($data->is_cancelled=='1'){
                            //assign_quantity adding new column (switch_supplier_id) 
                            $cancelled_by=\App\Models\User::findorfail($data->is_cancelled_by)->name;
                            
                            
                            $cancel_html='<h4><span class="badge badge-danger">Cancelled | Date : '.date('d-m-Y',strtotime($data->is_cancelled_date)).' | Cancel by : '.$cancelled_by.'</span></h4>';
                            
                        }else{
                             $cancel_html='';
                        }
                        
                        if($data->switch_status=='1'){
                            //assign_quantity adding new column (switch_supplier_id) 
                            if($data->switch_supplier_id!=0){
                                $switch_by=\App\Models\User::findorfail($data->switch_by)->name;
                                $switch_supplier_id=\App\Models\Supplier::findorfail($data->switch_supplier_id)->supplier_name;
                            
                                $switch_html='<h4><span class="badge badge-info">Switch Supplier | Date : '.date('d-m-Y',strtotime($data->switch_date)).' | Switch By : '.$switch_by.' | Switch Supplier : '.$switch_supplier_id.'</span></h4>';
                            }else{
                                $switch_html='';
                            }
                        }else{
                             $switch_html='';
                        }
                        $product_detail='
                            <div class="row">
                                <div class="col-md-2">
                                    <a class="single_image" href="'.$url.'" target="_blank" data-featherlight="image">
                                        '.$img.'
                                    </a>
                                </div>
                                <div class="col-md-10">
                                    <span class="text-success">'.$data->product_name.'</span><br>
                                    <span class="text-primary text-sm-left">SKU : <strong>'.$data->variation_id.'</strong></span><br>
                                    <span class="text-danger text-sm-left">PO No : <strong>'.$data->po_id.'</strong></span><br>
                                    '.$switch_html.$cancel_html.'
                                </div>
                            </div>
                            
                        ';
                        return $product_detail;
                    })
                    ->filterColumn('variation_id',function($query,$keyword){
                        $query->orWhereRaw("LOWER(product_info.product_name) LIKE ?", ["%{$keyword}%"])
                                ->orWhereRaw("assign_quantity.variation_id LIKE ?", ["%{$keyword}%"])
                                ->orWhereRaw("assign_quantity.po_id LIKE ?", ["%{$keyword}%"]);     
                    })
                    ->orderColumn('variation_id', function ($query, $order) {
                        $query->orderBy('product_info.product_name', $order)
                            ->orderBy('assign_quantity.variation_id', $order);    
                    })
                    ->addColumn('receiving_input',function($data){
                        if($data->receiving_qty==$data->receive_qty){
                            $disabled='disabled';
                        }else{
                            $disabled='';
                        }
                        $hidden='
                            <input type="hidden" id="assignid_'.$data->id.'" name="id" value="'.$data->id.'"/>
                            <input type="hidden" id="variation_id_'.$data->id.'" name="id" value="'.$data->variation_id.'"/>
                            <input type="text" class="form-control form-control-sm has-success input_rec" id="receiving_qty_'.$data->id.'" name="receiving_qty" placeholder="0" '.$disabled.'/>
                            <span id="error_'.$data->id.'" class="text-danger text-sm" style="display:none"></span>';
                            
                        return $hidden;

                    })
                    ->addColumn('remaining', function($data){
                        $sub=(int)$data->receive_qty-(int)$data->receiving_qty;
                        
                        if($data->is_cancelled=='1'){
                            return 0;
                        }
                        
                        if($data->switch_status=='1'){
                            return 0;
                        }
                        return $sub;
                    })
                    ->addColumn('switch_qty', function($data){
                        $sub=(int)$data->receive_qty-(int)$data->receiving_qty;
                        if($data->switch_status=='1'){
                            $where=[
                                'variation_id' => $data->variation_id,
                                'pr_id' => $data->pr_id,
                                'supplier_id' => $data->switch_supplier_id
                            ];
                            $tb=\DB::table('assign_quantity')->where($where)->first();
                            // dd($tb);
                            
                            if($tb !=null){
                                $rem=$tb->receive_qty;
                                // dd($rem);
                                return $rem;
                                    
                            }
                            return $sub;
                            
                        }else{
                            return 0;
                        }
                        
                    })
                    ->addColumn('recieved_qty_new', function($data){
                        
                        if($data->switch_status=='1'){
                            $where=[
                                'variation_id' => $data->variation_id,
                                'pr_id' => $data->pr_id,
                                'supplier_id' => $data->switch_supplier_id
                            ];
                            $tb=\DB::table('assign_quantity')->where($where)->first();
                            
                            
                            if($tb !=null){
                                $rem=$tb->receive_qty;
                                // dd($rem);
                                return $rem;
                                    
                            }
                            return $data->receive_qty;
                            
                        }else{
                            return $data->receive_qty;
                        }
                        
                    })
                    ->addColumn('cancelled_qty', function($data){
                        $sub=(int)$data->receive_qty-(int)$data->receiving_qty;
                        if($data->is_cancelled=='1'){
                            return $sub;
                        }else{
                            return 0;
                        }
                        
                    })
                    ->addColumn('action', function($data){
                        if($data->receiving_qty==$data->receive_qty){
                            $receiving='';
                        }else{
                            $receiving='<button class="btn btn-info btn-sm received_btn" id="submit_'.$data->id.'"><i class="fab fa-get-pocket"></i> Received</button><br>';
                        }
                        $entry='';
                        // if($data->receiving_qty ==0){
                        //     $entry='';
                        // }else{
                        //     $entry='<button  class="btn btn-success btn-sm blister_btn mt-1" id="print_'.$data->id.'"><i class="fas fa-archive"></i>  Blister Entry</button> ';
                        // }
                        $barcodes=route('supplier.print.label',$data->id);
                        // return '<button class="btn btn-info btn-sm received_btn" id="submit_'.$data->id.'" onclick=receiving_update('.$data->id.')> Received</button>';
                        return $receiving.$entry;//

                    })
                    ->rawColumns(['variation_id','remaining','receiving_input','action','switch_qty','cancelled_qty','recieved_qty_new'])
                    ->make(true);
        }else{
            if(in_array($string,['receiving','complete'])){
                $suppliers=Supplier::latest()->where('id','!=',1)->where('id','!=','2')->get();
                if($string=='receiving'){
                    $po_list=\DB::table('assign_quantity')->select(['assign_quantity.po_id'])
                    ->where('assign_quantity.status',0)
                    ->where('assign_quantity.supplier_id','!=',1)->where('id','!=','2')
                    ->groupBy('assign_quantity.po_id')->get();
                }else if($string=='complete'){
                    $po_list=\DB::table('assign_quantity')->select(['assign_quantity.po_id'])
                    ->where('assign_quantity.status',1)
                    ->where('assign_quantity.supplier_id','!=',1)->where('id','!=','2')
                    ->groupBy('assign_quantity.po_id')->get();
                }
                
                return view('supplier.receiving',['suppliers' => $suppliers,'order_string' => $string,'po_list' => $po_list]);
            }else{
                return abort(404);
            }
            
        }
        
    }
    
    public function get_gatepass_serial(){
        $check =\DB::table('gatepass_serial')->get();
        $serial=1;
        if($check->count()>0){
            $row=$check->first();
            $today=strtotime(date("Y-m-d"));
            $update_date= strtotime($row->update_date);
            
            $datediff = $today - $update_date;
            $difference = floor($datediff/(60*60*24));
            if($difference==0){
                $serial=$row->gate_pass_serial;
            }else{
                $serial=(int)$row->gate_pass_serial+1;
               
                $update_re=[
                    'gate_pass_serial' => $serial,
                    'update_date' => date('Y-m-d')
                ];
                
                \DB::table('gatepass_serial')->update($update_re);
            }
        }else{
            $insert=[
                'gate_pass_serial' => $serial,
                'supplier_id' => 1,
                'created_date' => date('Y-m-d'),
                'update_date' => date('Y-m-d')
            ];
            
            \DB::table('gatepass_serial')->insert($insert);
            
        }
        return $serial;
    }

    public function receive_input(Request $request){
        $input = $request->all();
        // dd($input);
        $approve=\DB::table('assign_quantity')->where('id', $input['id'])->first();
        $sku=Sku::where('variation_id',$input['variation_id'])->first();
        $order=\DB::table('view_products_orders')->where('variation_id',$input['variation_id'])->where('pr_id',$approve->pr_id)->first();
        $receiving_qty=(int)$input['old_rec']+$input['receive_qty'];
        
        if($approve->status==1){
            return response()->json([
                'receive_qty'=> $input['receive_qty'],
                'response' => '0',
                'code' => '400',
                'message' => 'Already Drop In'
            ]);
        }
        //added by sufian
        $gatepass_serial=$this->get_gatepass_serial();
        
        
        if($approve->supplier_id =='1'){
            $qty=$sku->Quantity;
            $qty=(int)$qty-$input['receive_qty'];
            Sku::where('variation_id',$input['variation_id'])->update(['Quantity' => ($qty<0)?0:$qty]);

            

        }else{
            $qty=$sku->Quantity;
            $qty=(int)$qty+(int)$input['receive_qty'];
            Sku::where('variation_id',$input['variation_id'])->update(['Quantity' => ($qty<0)?0:$qty]);
        }

        $insert_log=[
            'variation_id' => $input['variation_id'],
            'pr_id' => $approve->pr_id,
            'given_qty' => $input['receive_qty'],
            'supplier_id' =>$approve->supplier_id,
            'user_id' => auth()->user()->id,
            'type_name' => 'Order Received',
            'created_date' => date('Y-m-d H:i:s')
        ];

        $insert_data=\DB::table('item_log')->insert($insert_log);

        $insert_last=[
            'order_id' => $order->id,
            'assign_id' => $input['id'],
            'variation_id' => $input['variation_id'],
            'last_receiving' => $input['receive_qty'],
            'supplier_id' => $approve->supplier_id,
            'pr_id' => $approve->pr_id,
            'user_id' => auth()->user()->id,
            'created_date' => date('Y-m-d H:i:s'),
            'po_id' => $approve->po_id,
            'rec_id' => $approve->po_id.'-REC'.$gatepass_serial,
            'gatepass_serial' => $gatepass_serial
        ];

        $last_receiving_id=\DB::table('product_receiving')->insertGetId($insert_last);
        
        if($approve->not_to_wh==0){
            $insert22=[
                'variation_id' => $input['variation_id'],
                'variation_design' => $input['variation_id'],
                'main_type' => $approve->main_type,
                'prod_size' => $approve->prod_size,
                'in_stock' =>  $approve->in_stock,
                'supplier_id' => 1,
                'assign_date' => date('Y-m-d'),
                'order_qty_given' => $input['receive_qty'],
                'receive_qty' => $input['receive_qty'],
                'pr_id' => $approve->pr_id,
                'po_id' => $approve->po_id.'-WH',
                'last_receiving_id' =>$last_receiving_id,
            ];
            // $data[]=$insert;
            $response=\DB::table('assign_quantity')->insert($insert22);
            
            $insert_log=[
                'variation_id' => $input['variation_id'],
                'pr_id' => $approve->pr_id,
                'given_qty' => $input['receive_qty'],
                'supplier_id' => 1,
                'user_id' => auth()->user()->id,
                'type_name' => 'Assign Warehouse',
                'created_date' => date('Y-m-d H:i:s')
            ];
            $insert_data=\DB::table('item_log')->insert($insert_log);
        }
        
        
            
        

        $response=\DB::table('assign_quantity')
                ->where('id', $input['id'])
                ->update([
                    'receiving_qty' => $receiving_qty,
                    'received_at' => date('Y-m-d H:i:s'),
                ]);
        if($approve->receive_qty ==$receiving_qty){
            $response=\DB::table('assign_quantity')
                ->where('id', $input['id'])
                ->update(['status' => 1]);
        }
        return response()->json([
            'receive_qty'=> $input['receive_qty'],
            'response' => $response,
            'code' => '200',
            'message' => 'Successfully Drop IN'
        ]);
    }

    public function last_receiving(Request $request){

        $input=$request->all();
        if(isset($input['receiving_id']) !=''){
            $where=['id' => $input['receiving_id']];
        }else{
            $approve=\DB::table('assign_quantity')->where('id', $input['id'])->first();
            $order=\DB::table('view_products_orders')->where('variation_id',$input['variation_id'])->where('pr_id',$approve->pr_id)->first();
            $where=[
                'order_id' => $order->id,
                'assign_id' => $input['id'],
                'variation_id' => $input['variation_id'],
                'pr_id' => $approve->pr_id,
            ];
        }
        

        $record=\DB::table('product_receiving')->latest('id')->where($where)->first();
        
        return response()->json($record);
    }
    
    public function check_label(Request $request){
        $blis_qty=$request->b_qty;
        $label=$request->no_of_label;
        $data=\DB::table('product_receiving')->where('id',$request->id)->first();
        if($data->barcode_id !=0){
            $b_qty=\DB::table('barcodes')->where('id',$data->barcode_id)->first('blister_qty');
            $no_of_label=\DB::table('product_barcodes')->where('barcode_id',$data->barcode_id)->count();
            if($blis_qty==$b_qty->blister_qty && $label==$no_of_label){
                return response()->json([
                    'same' => '1',
                    'type' => 'same'
                ]);
            }else{
                return response()->json([
                    'same' => '0',
                    'type' => 'change'
                ]);
            }
        }else{
           return response()->json([
                'same' => '0',
                'type' => 'change'
            ]); 
        }
    }

    public function print_label($id,Request $request){
        $input = $request->all();

        if(isset($input['type'])){
            $where1=['id' => $id];
            $type=$id;
            $record=\DB::table('product_receiving')->latest('id')->where($where1)->first();
            $id=$record->assign_id;
        }
        $data=\DB::table('assign_quantity')->where('id', $id)->first();
        $product=\DB::table('product_info')->where('variation_id',$data->variation_id)
            ->join('teams','product_info.team','=','teams.team_code')
            ->first();
        $supplier=\App\Models\Supplier::where('id',$data->supplier_id)->first();
        $words =$supplier->supplier_name;
        $order=\DB::table('view_products_orders')->where('variation_id',$data->variation_id)->where('pr_id',$data->pr_id)->first();
        // $acronym = $words[0].$words[1];
        $acronym = $supplier->supplier_name;
        $hash=\Str::random(8);

        $insert_log=[
            'variation_id' => $data->variation_id,
            'pr_id' => $data->pr_id,
            'given_qty' => $input['receive_qty'],
            'supplier_id' =>$data->supplier_id,
            'user_id' => auth()->user()->id,
            'type_name' => 'Print Label',
            'created_date' => date('Y-m-d H:i:s')
        ];

        $insert_data=\DB::table('item_log')->insert($insert_log);

        
        $returnHTML=view('print.barcodes',[
                                            'data' => $data,
                                            'no_of_label' => $input['no_of_label'],
                                            'product'=>$product,
                                            'order'=> $order,
                                            'acronym' =>$acronym,
                                            'hash' => $hash,
                                            'insert'=>1,
                                            'type' => isset($input['type']) ? $type:'',
                                            'receive' => $input['receive_qty']])->render();
        // return view('print.barcodes',[
        //     'data' => $data,
        //     'no_of_label' => 5,
        //     'product'=>$product,
        //     'order'=> $order,
        //     'acronym' =>$acronym,
        //     'hash' => $hash,
        //     // 'insert'=>1,
        //     'receive' => 15]);

        return response()->json( array('success' => true, 'html'=>$returnHTML));
    }

    public function print_label2(Request $request){
        $input = $request->all();
        $where=['id' => $input['id']];
        $record=\DB::table('product_receiving')->where($where)->first();

        $product=\DB::table('product_info')->where('variation_id',$record->variation_id)
            ->join('teams','product_info.team','=','teams.team_code')
            ->first();

        $supplier=Supplier::where('id',$record->supplier_id)->first();
        $words =$supplier->supplier_name;
        $order=\DB::table('view_products_orders')->where('variation_id',$record->variation_id)->where('pr_id',$record->pr_id)->first();
        // $acronym = $words[0].$words[1];
        $acronym = $supplier->supplier_name;
        $hash=\Str::random(8);

        $insert_log=[
            'variation_id' => $record->variation_id,
            'pr_id' => $record->pr_id,
            'given_qty' => $record->last_receiving,
            'supplier_id' =>$record->supplier_id,
            'user_id' => auth()->user()->id,
            'type_name' => 'Again Print Label',
            'created_date' => date('Y-m-d H:i:s')
        ];
        $insert_data=\DB::table('item_log')->insert($insert_log);

        $returnHTML=view('print.barcodes',[
            'data' => $record,
            'product'=>$product,
            'order'=> $order,
            'acronym' =>$acronym,
            'hash' => $hash,
            'type' => '',
            'receive' => $record->last_receiving])->render();

        return response()->json( array('success' => true, 'html'=>$returnHTML));
    }



    public function print_label_list(Request $request){
        if($request->ajax()){
            $supplier=$request->supplier_id;
            $records=\DB::table('product_receiving')->where('product_receiving.supplier_id',$supplier)
            ->join('product_info','product_info.variation_id','=','product_receiving.variation_id')
            ->join('product_size','product_size.id','=','product_info.prod_size')
            ->LeftJoin('users','users.id','=','product_receiving.user_id')
            ->LeftJoin('suppliers','suppliers.id','=','product_receiving.supplier_id')
            ->select([
                'product_receiving.last_receiving as receive_qtn',
                'product_receiving.*',
                'product_info.prod_img',
                'product_info.product_url',
                'product_info.product_name',
                'product_size.size',
                'users.name',
                'suppliers.supplier_name',
            ]);

            return Datatables::of($records)
                    ->addIndexColumn()
                    ->addColumn('productDetail',function($data){
                        if($data->product_url !=''){
                            $url=$data->product_url;
                            $img=' <img src="'.$data->product_url.'" class="rounded img-thumbnail" alt="Cinque Terre" width="100" height="100">';
                        }else{
                            $url=asset('uploads/sku_images/'.$data->prod_img);
                            $img='<img src="'.asset('uploads/sku_images/'.$data->prod_img).'" class="rounded img-thumbnail" alt="Cinque Terre" width="100" height="100">';
                        }

                        $detail='
                        <div class="row">
                            <div class="col-md-2">
                                <a class="single_image" href="'.$url.'" target="_blank" data-featherlight="image">
                                    '.$img.'
                                </a>
                            </div>
                            <div class="col-md-10">
                                '.$data->product_name.'<br>
                                <span class="text-success">SKU : <strong>'.$data->variation_id.'</strong></span><br>
                                <span class="text-primary">Size : <strong>'.$data->size.'</strong></span><br>
                                <span style="color:#D35400">PR ID : <strong>'.$data->pr_id.'</strong></span>
                            </div>
                        </div>';

                        return $detail;
                    })
                    ->filterColumn('productDetail',function($query,$keyword){
                        $query->orWhereRaw("product_receiving.variation_id LIKE ?", ["%{$keyword}%"]);
                        $query->orWhereRaw("product_receiving.pr_id LIKE ?", ["%{$keyword}%"]);
                        $query->orWhereRaw("product_info.product_name LIKE ?", ["%{$keyword}%"]);
                    })
                    ->orderColumn('productDetail', function ($query, $order) {
                        $query->orderBy('product_receiving.variation_id', $order);
                        $query->orderBy('product_receiving.pr_id', $order);
                        $query->orderBy('product_info.product_name', $order);
                    })
                    ->addColumn('created_at', function($data){
                        return date('d-m-Y',strtotime($data->created_date));
                     })
                    ->filterColumn('created_at',function($query,$keyword){
                        $keyword=date('Y-m-d',strtotime($keyword));
                        $query->orWhereRaw("product_receiving.created_date LIKE ?", ["%{$keyword}%"]);
                    })
                    ->orderColumn('created_at', function ($query, $order) {
                        $query->orderBy('product_receiving.created_date', $order);
                    })
                    ->addColumn('action', function($data){
                        $entry='<button  class="btn btn-success btn-sm blister_btn mt-1" id="print_'.$data->id.'"><i class="fas fa-archive"></i>  Print Label</button> ';
                        return $entry;//

                    })
                    ->addColumn('receiving',function($data){
                        if($data->barcode_id !=0){
                            $b_qty=\DB::table('barcodes')->where('id',$data->barcode_id)->first('blister_qty');
                            $no_of_label=\DB::table('product_barcodes')->where('barcode_id',$data->barcode_id)->count();
                            $blister_text='<span class="text-danger">Blister QTY : <strong id="b_qty_'.$data->id.'">'.$b_qty->blister_qty.'</strong></span><br>
                            <span class="text-info">No of label : <strong id="label_'.$data->id.'">'.$no_of_label.'</strong></span>';
                            
                        }else{
                            $blister_text='<span class="text-danger">Blister QTY : <strong>0</strong></span><br>
                            <span class="text-info">No of label : <strong>0</strong></span>';
                        }
                        return $detail='<span>'.$data->receive_qtn.'</span><br>'.$blister_text;
                    })
                    ->rawColumns(['created_at','productDetail','action','receiving'])
                    ->make(true);

        }else{
            $suppliers=Supplier::latest()->get();
            return view('supplier.printing',['suppliers' => $suppliers]);
            
        }

    }
    
    public function make_pending_pdf(Request $request){
        $supplier=$request->supplier_id;
        $is_print=$request->is_print;
        $po_id=$request->po_id2;
        if($supplier !='ALL'){
            $supplier_data=Supplier::where('id' , $supplier)->first('supplier_name');
            $supplier_name=$supplier_data->supplier_name;
        }else{
            $supplier_name='ALL';
        }
        $records=\DB::table('assign_quantity')
                    ->join('suppliers','suppliers.id','=','assign_quantity.supplier_id')
                    ->join('product_info','product_info.variation_id','=','assign_quantity.variation_id')
                    ->join('product_size','assign_quantity.prod_size','=','product_size.id')
                    ->select([
                        'assign_quantity.*',
                        'suppliers.supplier_name',
                        'product_info.prod_img',
                        'product_size.size',
                        'product_info.product_name',
                        'product_info.product_url'
                    ])
                    
                    ->where('assign_quantity.status',0);
        
        if($supplier != 'ALL'){
            $records->where('assign_quantity.supplier_id',$supplier);
        }
        
        if($po_id !='ALL'){
            $po_id_a=explode(',',$po_id);
            if($po_id_a[0]==''){
                $po_idnew=\DB::table('assign_quantity')
                    ->groupBy('assign_quantity.po_id')
                    ->select([
                        'assign_quantity.po_id',
                    ])
                    ->where('assign_quantity.status',0)
                    ->where('assign_quantity.supplier_id',$supplier)
                    ->pluck('po_id')
                    ->toArray();
                    
                $po_id=implode(',',$po_idnew);
            }else{
                $records->whereIn('assign_quantity.po_id',$po_id_a);
            }
            
            // $po_id= $po_id_a;
        }else{
            $po_idnew=\DB::table('assign_quantity')
                    ->groupBy('assign_quantity.po_id')
                    ->select([
                        'assign_quantity.po_id',
                    ])
                    ->where('assign_quantity.status',0)
                    ->where('assign_quantity.supplier_id',$supplier)
                    ->pluck('po_id')
                    ->toArray();
                    
            $po_id=implode(',',$po_idnew);
            // dd();
        }
        
        if($is_print !=''){
            $html= view('pdf.pending_orders',[
                'records' => $records->get(),
                'supplier_name' => $supplier_name,
                'po_id' => $po_id, 
                ])->render();
            return response()->json(['success'=>true,'html' => $html]);
        }else{
            $pdf = PDF::loadView('pdf.pending_orders', [
                'records' => $records->get(),
                'supplier_name' => $supplier_name,
                'po_id' => $po_id,
            ])->setPaper('A4', 'protrait');
    
            return $pdf->download('purchase_order_'.date('d_m_Y').'.pdf');
        }
        
        
    }
    
    public function po_list(Request $request){
        if($request->ajax()){
            $supplier =$request->supplier_id;
            $saletype=$request->sale_type;
            $records=\DB::table('assign_quantity')
                    ->join('suppliers','suppliers.id','=','assign_quantity.supplier_id')
                    ->select(['assign_quantity.po_id','suppliers.supplier_name','assign_quantity.assign_date'])
                    
                    ->groupBy('assign_quantity.po_id');
            $total_po=\DB::table('assign_quantity')
                    ->select([
                            \DB::raw('sum(assign_quantity.receive_qty) as total_order,sum(assign_quantity.receiving_qty) as total_received,(sum(assign_quantity.receive_qty)-sum(assign_quantity.receiving_qty)) as remaining')
                        ])
                    ->where('assign_quantity.supplier_id','!=',1)
                        ->where('assign_quantity.supplier_id','!=',2);
            if($supplier !=''){
                $records->where('assign_quantity.supplier_id',$supplier); 
                $total_po->where('assign_quantity.supplier_id',$supplier);
                
            }
            
            if($saletype=='complete'){
                $records->where('assign_quantity.status',1);
                // $total_po->where('assign_quantity.status',1);
                
                $assign=\DB::table('assign_quantity')
                ->where('supplier_id',$supplier)
                // ->where('pr_id',$pr_id)
                ->where('supplier_id','!=',1)
                ->where('supplier_id','!=',2)
                ->where('switch_status','0')
                ->select([
                    \DB::raw('sum(assign_quantity.order_qty_given) as order_qty_given'),
                    \DB::raw('sum(assign_quantity.receive_qty) as receive_qty'),
                    \DB::raw('sum(assign_quantity.receiving_qty) as receiving_qty'),
                    'switch_status',
                ])->first();
                
            $assign2=\DB::table('assign_quantity')
                ->where('supplier_id',$supplier)
                // ->where('pr_id',$pr_id)
                ->where('supplier_id','!=',1)
                ->where('supplier_id','!=',2)
                ->where('switch_status','1')
                ->where('receiving_qty','!=','0')
                ->select([
                    \DB::raw('sum(assign_quantity.order_qty_given) as order_qty_given'),  
                    \DB::raw('sum(assign_quantity.receive_qty) as receive_qty'),
                    \DB::raw('sum(assign_quantity.receiving_qty) as receiving_qty'),
                    'switch_status',
                ])->first();
            
            $total_order2=(int)($assign->order_qty_given !="")?$assign->order_qty_given:0;
            $total_order=(int)($assign->receive_qty !="")?$assign->receive_qty:0;
            $total_receive_qty=($assign->receiving_qty!="")?$assign->receiving_qty:0;
            
            $switch_order_qty=(int)($assign2->order_qty_given !="")?$assign2->order_qty_given:0;
            $switch_order=(int)($assign2->receive_qty !="")?$assign2->receive_qty:0;
            $switch_receive_qty=($assign2->receiving_qty!="")?$assign2->receiving_qty:0;
            
            $total_order+=$switch_receive_qty;
            $total_receive_qty+=$switch_receive_qty;
            }else{
                $records->where('assign_quantity.status',0);
                $total_po->where('assign_quantity.status',0);
                
            }
            
            $html='<option value="">ALL</option>';
            foreach ($records->get()  as $record){
                $html.='<option value="'.$record->po_id.'">PO #'.$record->po_id.'&nbsp;&nbsp;|&nbsp;&nbsp; Supplier #: '.$record->supplier_name.'&nbsp;&nbsp;|&nbsp;&nbsp; Assign Date : '.date('d-m-Y',strtotime($record->assign_date)).'</option>';
            }
            if($saletype=='complete'){
                $total_order=$total_order;
                $total_received=$total_receive_qty;
                $remaining=0;
            }else{
                $totalPO=$total_po->first();
                $total_order=isset($totalPO->total_order)?$totalPO->total_order:0;
                $total_received=isset($totalPO->total_received)?$totalPO->total_received:0;
                $remaining=isset($totalPO->remaining)?$totalPO->remaining:0;  
            }
            
            
            
            return response()->json(['success' => 1,'html' =>$html,'total_order' => $total_order,'total_received' => $total_received,'remaining' => $remaining]);
        }
    }
    
    public function get_sup_po_no($supplier){
        $check=\DB::table('suppliers')->where('id',$supplier)->get();
        $ponumber='';
        if($check->count()>0){
            $row=$check->first();
            $today=strtotime(date("Y-m-d"));
            $update_date= strtotime($row->sup_po_no_date);
            
            $datediff = $today - $update_date;
            $difference = floor($datediff/(60*60*24));
            if($difference==0){
                $serial=$row->sup_po_no;
                $ponumber=$row->short_name.$serial;
            }else{
                $serial=(int)$row->sup_po_no+1;
                $ponumber=$row->short_name.$serial;
               
                $update_re=[
                    'sup_po_no' => $serial,
                    'sup_po_no_date' => date('Y-m-d')
                ];
                
                \DB::table('suppliers')->where('id',$supplier)->update($update_re);
            }
        }
        return $ponumber;
    }
    
    
    public function switch_supplier(Request $request){
        if($request->ajax()){
            // dd($request->all());
            $assign_id=$request->assign_id;
            $actual_qty=$request->actual_qty;
            $assign_qty=$request->assign_qty;
            
            $check=\DB::table('assign_quantity')->where('id',$assign_id)->get();
            
            // dd($check);
            
            if($check->count()>0){
                $value=$check->first();
                
                $where=[
                    'variation_id' => $value->variation_id,
                ];
                
                $sku=\DB::table('product_info')->where($where)->first();
                
                
                $supplier=$request->switch_supplier_id;
                $po_serial=$this->get_sup_po_no($supplier);
                $po_id=$value->pr_id.'-'.$po_serial;
                
                
                $insert=[
                    'variation_id' => $value->variation_id,
                    'variation_design' => $value->variation_id,
                    'main_type' => $sku->group_id,
                    'prod_size' => $sku->prod_size,
                    'in_stock' =>  $value->in_stock,
                    'supplier_id' => $supplier,
                    'assign_date' => date('Y-m-d'),
                    'order_qty_given' => $assign_qty,
                    'receive_qty' => $assign_qty,
                    'pr_id' => $value->pr_id,
                    'po_id' => $po_id,
                    'old_assign_id' => $assign_id,
                    'old_supplier_id' => $value->supplier_id
                ];
                
                
                
                // echo "<pre>";print_r($insert);
            
                $response=\DB::table('assign_quantity')->insert($insert);
            
                $insert_log=[
                    'variation_id' => $value->variation_id,
                    'pr_id' => $value->pr_id,
                    'given_qty' =>$assign_qty,
                    'supplier_id' =>$supplier,
                    'user_id' => auth()->user()->id,
                    'type_name' => 'Switch Supplier',
                    'created_date' => date('Y-m-d H:i:s')
                ];
                
                // echo "<pre>";print_r($insert_log);
                
                $insert_data=\DB::table('item_log')->insert($insert_log);
                
                $rem=(int)$actual_qty-(int)$assign_qty;
                // dd($rem);
                
                if($rem >0){
                    $supplier=$value->supplier_id;
                    $po_serial=$this->get_sup_po_no($supplier);
                    $po_id=$value->pr_id.'-'.$po_serial;
                    
                    $insert=[
                        'variation_id' => $value->variation_id,
                        'variation_design' => $value->variation_id,
                        'main_type' => $sku->group_id,
                        'prod_size' => $sku->prod_size,
                        'in_stock' =>  $value->in_stock,
                        'supplier_id' => $supplier,
                        'assign_date' => date('Y-m-d'),
                        'order_qty_given' => $rem,
                        'receive_qty' => $rem,
                        'pr_id' => $value->pr_id,
                        'po_id' => $po_id,
                        //'old_assign_id' => $assign_id,
                        //'old_supplier_id' => $value->supplier_id
                    ];
                    // echo "<pre>";print_r($insert);
                    
                    $response=\DB::table('assign_quantity')->insert($insert);
                    
                    $insert_log=[
                        'variation_id' => $value->variation_id,
                        'pr_id' => $value->pr_id,
                        'given_qty' =>$rem,
                        'supplier_id' =>$supplier,
                        'user_id' => auth()->user()->id,
                        'type_name' => 'Switch Supplier',
                        'created_date' => date('Y-m-d H:i:s')
                    ];
                    // echo "<pre>";print_r($insert_log);
                    
                    $insert_data=\DB::table('item_log')->insert($insert_log);
                }
                
                $update_arr=[
                    'switch_date' => date('Y-m-d H:i:s'),
                    'switch_status' => 1,
                    'status' => 1,
                    'comments' => $request->comments,
                    'switch_by' => auth()->user()->id,
                    'switch_supplier_id' => $request->switch_supplier_id
                ];
                // echo "<pre>";print_r($update_arr);
                
                \DB::table('assign_quantity')->where('id',$assign_id)->update($update_arr);
            }
        }
        
        return response()->json(['success' => 1,'message' => 'Supplier Switch Sucess fully against SKU']);
    }
    
    public function cancellation_supplier(Request $request){
        if($request->ajax()){
            // dd($request->all());
            $assign_id=$request->id;
            $comments=$request->comments;
            
            $check=\DB::table('assign_quantity')->where('id',$assign_id)->get();
            
            
            if($check->count()>0){
                $value=$check->first();
                
                $update_arr=[
                    'status' => 1,
                    'is_cancelled' => 1,
                    'is_cancelled_by' => auth()->user()->id,
                    'is_cancelled_date' => date('Y-m-d'),
                    'comments' => $comments
                ];
                
                \DB::table('assign_quantity')->where('id',$assign_id)->update($update_arr);
            }
        }
        
        return response()->json(['success' => 1,'message' => 'Supplier Order Cancelled Successfully']);
    }


    
}
